#!/bin/sh
# Run `wish' using the user's $PATH \
    exec wish -f "$0" ${1+"$@"}

# tkwallpaper.tcl:
# A TclTk script for auto-changing the background/root window image.
# Copyright (C) 1999  Maciej Kalisiak

# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.


################### NOTES #########################
#
# - 'cur_playlist' is a full path filename!
#


############################################################
## procedures

## prints $str to the status line (which times out in $sl_time_out ms)
proc sl_puts {str} {
  global status_line_text sl_timer_id

  ## constant
  set sl_time_out 4000
  
  ## kill any previous timer
  if {[info exists sl_timer_id]} {
    after cancel $sl_timer_id
  }

  set status_line_text $str

  ## set up the timer
  set sl_timer_id [after $sl_time_out [list set status_line_text ""]]
}


#
# given a full path filename, figures out its "nickname"
#
proc nickname {playlist} {
  return [file root [file tail $playlist]]
}


#
# reads the configuration file and sets appropriate run time variables
#
proc read_config {cfgfile} {
  global delay_min random_mode cur_playlist config_dir

  ## set 'default' as the default playlist, possibly overriden by config file
  set cur_playlist "$config_dir/default.playlist"

  if [catch {open $cfgfile r} file_id] {
    ## create a brand spanking new config file
    puts "Generating default config file \"$cfgfile\"."
    save_config $cfgfile
    create_playlist $cur_playlist
  } else {
    while {![eof $file_id]} {
      gets $file_id line
      if {[llength $line] == 0} {	;# skip blank lines
	continue
      }
      switch [lindex $line 0] {
	delay {
	  set delay_min [lindex $line 1]
	  delay_updated
	}
	mode {
	  if {[lindex $line 1] == "random"} {
	    set random_mode 1
	  } else {			;# TODO: what if more than 2 modes?
	    set radnom_mode 0
	  }
	}
	playlist {
	  set cur_playlist [lindex $line 1]
	}
	default {
	  puts "Strange line in config file: \]$line\["
	}
      }
    }
    close $file_id
  }

  load_playlist $cur_playlist

  sl_puts "Configuration loaded."
}


#
# save the current parameters into the configuration file
#
proc save_config {cfgfile} {
  global delay_min random_mode cur_playlist config_dir

  if {![file exists $config_dir]} {
    file mkdir $config_dir
  }
  
  if [catch {open $cfgfile w} file_id] {
    puts "ERROR: could not open '$cfgfile' for writing."
    exit
  }

  puts $file_id "delay $delay_min"
  if {$random_mode == 1} {
    puts $file_id "mode random"
  } else {
    puts $file_id "mode linear"
  }
  puts $file_id "playlist $cur_playlist"

  close $file_id

  sl_puts "Configuration saved."
}


#
# loads the given playlist (the filename contains full path)
#
proc load_playlist {playlist} {
  global files_list delay_min random_mode index

  if [catch {open $playlist r} file_id] {
    puts "Playlist '[nickname $playlist]' does not exist; creating it."
    create_playlist $playlist
  } else {
    set files_list {}			;# empty out the `files_list'
    while {![eof $file_id]} {
      gets $file_id line
      if {[llength $line] == 0} {	;# skip blank lines
	continue
      }
      switch [lindex $line 0] {
	file {
	  lappend files_list [lindex $line 1]
	}
	default {
	  puts "Strange line in playlist: \]$line\["
	}
      }
    }
    close $file_id
    reset_timer
    set index 0
  }

  sl_puts "Playlist '[nickname $playlist]' loaded."
}


#
# saves 'cur_playlist'
#
proc save_playlist {} {
  global files_list cur_playlist

  ## TODO: we should 'catch' any exceptions thrown by the 'open'
  set file_id [open $cur_playlist w]
  foreach filename $files_list {
    puts $file_id "file $filename"
  }
  close $file_id

  sl_puts "Playlist '[nickname $cur_playlist]' saved."
}


#
# deletes 'cur_playlist'
#
proc del_playlist {} {
  global cur_playlist pl_menu

  ## TODO: prevent deletion of the `default' playlist
  
  file delete $cur_playlist

  ## TODO: should not be touching GUI (ewwwww!) stuff here!
  $pl_menu delete [nickname $cur_playlist]
  $pl_menu invoke 0			;# invoke the first playlist
}


#
# a simple dialog box which prompts for a 
# string input, which is then returned
#
proc get_data_dlg {prompt} {
  ## TODO: does it have to be global? callback scope problem I bet...
  global dlg_res
  
  toplevel .dlg
  message .dlg.msg -text $prompt
  entry .dlg.ent -textvariable val
  pack .dlg.msg -side top
  pack .dlg.ent -side top
  bind .dlg.ent <Return> {set dlg_res $val; destroy .dlg}

  focus .dlg.ent
  grab .dlg
  tkwait window .dlg
  grab release .dlg

  return $dlg_res
}


#
# asks the user for a new playlist nickname, and then calls create_playlist
#
proc new_playlist {} {
  global config_dir
  
  # get the name for a new playlist
  set name [get_data_dlg "Name of new playlist:"]

  # TODO: we should check that no error occurred
  create_playlist $config_dir/$name.playlist
}
  

#
# given a playlist full path and filename, tries to create it
#
proc create_playlist {playlist} {
  global cur_playlist files_list pl_menu cur_playlist_short

  set nick [nickname $playlist]

  # check that it does not exist
  ## TODO
  
  # set `cur_playlist' to this name
  set cur_playlist $playlist
  
  # perform `save_playlist' with an empty `files_list'
  set files_list {}
  save_playlist

  # put this playlist into the playlist_menu
  ## TODO: we don't want to be doing GUI stuff here...
  ## NOTE: we don't put in 'default' as it will be already there
  ##       from GUI init code
  if {[string compare $nick "default"] != 0} {
    $pl_menu add radio -label $nick -variable cur_playlist_short\
	-command playlist_changed
  }

  sl_puts "Playlist '$nick' created."
}


#
# returns a list of all currently existing playlists
#
proc list_playlists {} {
  global config_dir
  
  set result {}
  if {[file exists $config_dir]} {
    set full_playlists [glob "$config_dir/*.playlist"]
    foreach pl $full_playlists {
      lappend result [nickname $pl]
    }
  }
  return $result
}


#
# converts a large "seconds" count into corresponding hours, mins, and secs;
# the result is a nicely formatted string
#
proc secs_to_time_display {secs} {
  set hours [expr $secs / 3600]
  set secs [expr $secs - $hours*60]
  set mins [expr $secs / 60]
  set secs [expr $secs - $mins*60]

  if {$hours > 0} {
    return [format "%2d:%02d:%02d" $hours $mins $secs]
  } elseif {$mins > 0} {
    return [format "%02d:%02d" $mins $secs]
  } else {
    return [format "0:%02d" $secs]    
  }
}


#
# resets the background refresh timer
#
proc reset_timer {} {
  global secs_left
  set secs_left 1			;# not 0 since this value will get
					;# decremented right away
}
  

#
# format `$number' (which is numeric, not string) to have a comma
# every three digits
#
proc put_in_commas {number} {
  set str [format "%d" $number]
  set result [string index $str 0]
  set idx 1;
  for {set i [expr [string length $str]-1]} {$i > 0} {incr i -1; incr idx} {
    if {[expr $i % 3] == 0} {
      set result "$result,[string index $str $idx]"
    } else {
      set result "$result[string index $str $idx]"
    }
  }
  return $result
}


#
# given a full path filename, generates an appropriate string to be put
# into the listbox
#
proc gen_line {full_file} {
  set file_w 25				;# the width of the filename field
  set file [file tail $full_file]
  if {[string length $file] > $file_w} {
    set file [string range $file 0 [expr $file_w-1]]
  }

  set size_k [expr [file size $full_file] / 1024]
  return [format "%-${file_w}s|%5dk" $file $size_k]
}


#
# empty the listbox, then fill it with entries based on 'files_list'
#
proc fill_listbox {} {
  global lb random_mode files_list listbox_map

  # first empty out the listbox and the map
  $lb delete 0 end
  set listbox_map {}
  
  # generate a linear, sequential mapping
  for {set i 0} {$i < [llength $files_list]} {incr i} {
    lappend listbox_map $i
  }

  # if random_mode is on, jumble the mapping
  if {$random_mode} {
    set listbox_map [jumble_list $listbox_map]
  }

  # now fill'er up
  for {set i 0} {$i < [llength $listbox_map]} {incr i} {
    set full_file [lindex $files_list [lindex $listbox_map $i]]
    $lb insert end [gen_line $full_file]
  }
}


#
# given a listbox line index, this fn returns the full filename to that file
#
proc index_to_filename {idx} {
  global files_list listbox_map
  return [lindex $files_list [lindex $listbox_map $idx]]
}


#
# does the inverse mapping, i.e., from full filename to index
# returns '-1' if file not found in $files_list
#
proc filename_to_index {filename} {
  global files_list listbox_map

  set files_list_index [lsearch $files_list $filename]
  if {$files_list_index == -1} {
    return -1
  }
  return [lsearch $listbox_map $files_list_index]
}


#
# returns a new list, which is a randomly jumbled version of the one passed in
#
proc jumble_list {orig_list} {
  set rand_list {}
  while {[llength $orig_list] > 0} {
    set rand_num [random_range [llength $orig_list]]
    lappend rand_list [lindex $orig_list $rand_num]
    set orig_list [lreplace $orig_list $rand_num $rand_num]
  }
  return $rand_list
}


#
# toss pics from $files_list based on the listbox's 'curselection'
#
proc remove_pics {} {
  global lb files_list listbox_map index

  # save the full name of current image
  set cur_image [index_to_filename $index]
  
  foreach i [lsort -decreasing [$lb curselection]] {
    set actual_index [lindex $listbox_map $i]
    set files_list [lreplace $files_list $actual_index $actual_index]
  }

  fill_listbox

  # now point at the new image again or reset the index
  set index [filename_to_index $cur_image]
  if {$index < 0} {
    set index 0
    show_image [index_to_filename 0]
  }

  # make the listbox point at the file
  follow_sel

  sl_puts "Image(s) removed."
}


#
# add pics to image listbox
#
proc add_pics {} {
  global lb

  $lb selection clear 0 end

  foreach file [ file_browser ] {
    $lb insert end $file
  }
}


#
# function called on exit
#
proc exit_func {} {
  ### TODO: it's only hooked up to "Quit"; hook up window close method also
  # for now a simple exit; later, save configuration, etc...
  exit
}


#
# show the image `$img' using `$viewer_cmd'
#
proc show_image {img} {
  global viewer_cmd

  sl_puts "Loading file: [file tail $img]"
  
  # run it in the background
  eval exec $viewer_cmd $img "&"

  wm title . "TkWallpaper: [file tail $img]"
  wm iconname . [file tail $img]
}


#
# point the selection bar at currently showing pic, and make sure it's
# visible in the listbox
#
proc follow_sel {} {
  global lb index sb

  $lb see $index
  $lb selection clear 0 end
  $lb selection set $index

  update				;# need this update for some reason
					;# TODO: why?
  # now make sure that the scrollbar is tracking it too
  eval $sb set [$lb yview]
}


#
# recalculates the delay into seconds; restarts the timer
#
proc delay_updated {} {
  global delay_min delay lock secs_left

  set delay [expr int(60 * $delay_min)]
  set secs_left $delay
}


#
# select the image under (x,y)
#
proc sel_image {x y} {
  global lb index image_lock

  set index [expr [$lb index @$x,$y] - 1]
  set image_lock 0			;# unlock
  reset_timer
}


#
# back up to the previous image
#
proc prev_image {} {
  global index lb

  if {$index == 0} {
    set index [expr [$lb index end] -1]
    puts $index
  } else {
    incr index -1
  }
  incr index -1				;# this cancels out the auto increment

  reset_timer
}


#
# advance to next image
#
proc next_image {} {
  global index lb

  if {$index == [expr [$lb index end] -1]} {
    set index 0
  } else {
    incr index
  }
  incr index -1				;# this cancels out the auto increment

  reset_timer
}


#
# init the PRNG
#
proc rand_init { seed } {
  global rand_seed
  set rand_seed $seed
}


#
# cheapo PRNG
# returns values in the range [0..1)
#
proc random {} {
  global rand_seed
  set rand_seed [expr ($rand_seed*9301 + 49297) % 233280]
  return [expr $rand_seed/double(233280)]
}


#
# returns a random value in the range [0..$range)
#
proc random_range {range} {
  expr int([random] * $range)
}


#
# toggles the random mode setting, and correspondingly updates the image list
#
proc toggle_random {} {
  global index

  # get the filename of the pic currently displayed
  set cur_filename [index_to_filename $index]
  
  # redo the listbox
  fill_listbox

  # now select the same picture (in its new listbox position)
  set index [filename_to_index $cur_filename]
  follow_sel
}


#
# configure all playlist_menu entries to invoke `playlist_changed' whenever
# a different playlist is selected
#
proc config_pl_menu {} {
  global pl_menu

  for {set i 0} {$i <= [$pl_menu index last]} {incr i} {
    $pl_menu entryconfigure $i -command playlist_changed
  }
}


#
# loads the newly selected playlist
#
proc playlist_changed {} {
  global cur_playlist cur_playlist_short config_dir
  set cur_playlist $config_dir/$cur_playlist_short.playlist
  load_playlist $cur_playlist
  # TODO: is this necessary?
  fill_listbox
}

#--------------------------------------------
# the file browser code

proc file_browser {} {
  global fb_lb fb_sb fb_diredit fb_grab fb_done files_font
  global Pics

  $Pics entryconfigure "Add" -state disabled

  toplevel .fb
  frame .fb.f
  set fb_lb [listbox .fb.f.list -yscroll {$fb_sb set} -width 40 -height 20\
		 -font $files_font -exportselection false]
  set fb_sb [scrollbar .fb.f.scroll -command "$fb_lb yview"]
  set fb_diredit [entry .fb.diredit -textvariable cur_dir -relief sunken]
  set fb_grab [button .fb.grab -text "Grab" -command grab_files]
  set fb_test [button .fb.test -text "View" -command test_image]
  set fb_done [button .fb.done -text "Done" -command done_adding_pics]

  pack $fb_sb -side right -fill y
  pack $fb_lb -side right 
  pack .fb.f -side top
  pack $fb_diredit -side top -fill x -expand yes -padx 5 -pady 3
  pack $fb_grab -side left 
  pack $fb_test -side left
  pack $fb_done -side right

  $fb_lb configure -selectmode extended

  bind $fb_lb <Double-1> {get_selection %x %y}
  bind $fb_diredit <Return> {read_dir}

  wm title .fb "Add Pics" 

  read_dir
}

proc read_dir {} {
  global fb_lb cur_dir

  $fb_lb delete 0 end

  set orig_pwd [pwd]
  cd $cur_dir
  set cur_dir [pwd]

  foreach i [lsort [glob * .*]] {
    if [file isdirectory $i] {
      set i $i/
    }
    if {[string compare $i "./"] != 0} {
      $fb_lb insert end $i
    }
  }

  cd $orig_pwd
}

proc get_selection {x y} {
  global fb_lb cur_dir fwdslash

  set item [$fb_lb get @$x,$y]
  if {[is_dir $item]} {
    set cur_dir [file join $cur_dir $item]
    read_dir
  } else {
    test_image
  }
}

proc grab_files {} {
  global fb_lb cur_dir

  foreach i [$fb_lb curselection] {
    set file [$fb_lb get $i]
    if {! [is_dir $file]} {
      set fullfile [file join $cur_dir $file]
      add_file_to_list $fullfile
    }
  }
  # have to remake it now since we got new entries
  fill_listbox
  $fb_lb selection clear 0 end

  sl_puts "Images added."
}

proc add_file_to_list {file} {
  global files_list
  lappend files_list $file
}

proc is_dir {name} {
  set lastindex [expr [string length $name] - 1]
  if {[expr {[string index $name $lastindex]} == {"/"}]} {
    return 1
  } else {
    return 0
  }
}

proc done_adding_pics {} {
  global Pics fb_lb

  if {[llength [$fb_lb curselection]] > 0} {
    set choice [tk_messageBox -type yesno -default yes -icon question\
		    -message "Grab the currently selected files?"]
    if {$choice == "yes"} {
      grab_files
    }
  }
  destroy .fb
  $Pics entryconfigure "Add" -state normal
}

proc test_image {} {
  global fb_lb cur_dir viewer_cmd

  set file [$fb_lb get [lindex [$fb_lb curselection] 0]]
  set fullfile [file join $cur_dir $file]
  eval exec $viewer_cmd $fullfile "&"

  sl_puts "Testing $file"
}



############################################################
## end of subroutines; start of global vars

set lib_files_dir "/usr/local/lib/tkwallpaper"
set viewer_cmd "nice xv -root -quit -max -fixed -rmode 5 +noresetroot"
#set viewer_cmd "nice xv -root -quit -fixed -rmode 5 +noresetroot"

## some defaults
set config_dir "$env(HOME)/.tkwallpaper"
set config_file "$config_dir/tkwallpaperrc"
set delay_min 30
set delay [expr 60 * $delay_min]
set image_lock 0
set random_mode 1
set rand_seed [pid]
set files_list {}
set cur_dir "."

## the `files_font' should be fixed (not proportionally-spaced)
set files_font "-*-courier-medium-r-*-*-12-*-*-*-*-*-*-*"
set status_font "-*-helvetica-medium-r-*-*-12-*-*-*-*-*-*-*"
set default_font "-*-helvetica-medium-r-*-*-12-*-*-*-*-*-*-*"

option add *font $default_font
option add *padY 0

##--------------------------
## geometry

frame .status
label .status.line -relief sunken -textvariable status_line_text -anchor w\
    -font $status_font -borderwidth 1
pack .status.line -fill x -padx 3 -pady 3
pack .status -side bottom -fill x

frame .fr_top
pack .fr_top -side top -fill x

label .fr_top.lb_time_left -text "Time left:"
label .fr_top.time_left -textvariable time_left
pack .fr_top.time_left -side right
pack .fr_top.lb_time_left -side right

label .fr_top.lb_playlist -text "Playlist:"
## this has to be `eval'ed so that the list of playlists gets broken up...
## NOTE: since we have to provide at least one item for the menu, we put
##       'default' in ahead of time
set pllist [list_playlists]
if {[llength $pllist] == 0} {
  set pllist [list default]
}
set pl_menu [eval "tk_optionMenu .fr_top.lists cur_playlist_short $pllist"]
config_pl_menu

pack .fr_top.lb_playlist -side left
pack .fr_top.lists -side left

frame .bot
label .bot.delay_lbl -text "Delay (min):" 
set dly [entry .bot.delay -textvariable delay_min -width 5 -borderwidth 1]
button .bot.prev -text Prev -command prev_image
button .bot.next -text Next -command next_image

pack .bot.delay_lbl -side left
pack .bot.delay -side left
pack .bot.next -side right
pack .bot.prev -side right
pack .bot -side bottom -anchor w -fill x


frame .flb
set lb [listbox .flb.list -yscroll {$sb set} -width 35 -height 20\
	    -selectmode extended]
set sb [scrollbar .flb.scroll -command {$lb yview}]

pack $lb -side left
pack $sb -side left -fill y
pack .flb -side left

## menubar
menu .menubar -borderwidth 1 -activeborderwidth 1
foreach m {Playlist Pics Options} {
  set $m [menu .menubar.m$m -tearoff false -borderwidth 1 -activeborderwidth 1]
  .menubar add cascade -label $m -menu .menubar.m$m
}

$Playlist add command -label "New" -command new_playlist
$Playlist add command -label "Save" -command save_playlist
$Playlist add command -label "Delete" -command del_playlist
$Playlist add separator
$Playlist add command -label "Quit" -command exit_func

$Pics add command -label "Add" -command add_pics
$Pics add command -label "Remove" -command remove_pics

$Options add check -label "Random" -variable random_mode \
    -command toggle_random
$Options add check -label "Lock" -variable image_lock
$Options add separator
$Options add command -label "Save config." -command "save_config $config_file"

## attach menubar to main window
. config -menu .menubar

##----------
## do some window manager stuff (title and iconname are changed dynamically
## elsewhere)
wm command . [concat $argv0 $argv]
wm group . .
wm iconbitmap . @$lib_files_dir/icon.xbm
#wm iconify .

##---------------------
## bindings 
bind $dly <Return> delay_updated
bind $lb <Double-1> {sel_image %x %y}

##-------------------------
## configuration section

## this is not a good thing: what if somebody does not have this font???
## should really use resources for this...
$lb configure -font $files_font
$lb configure -exportselection false

##------------ main()
# seed the random number generator with random number
rand_init [exec date +%s]

##--------------------------
## read configuration (or create it) ASAP
read_config $config_file

fill_listbox
$pl_menu invoke [file root [file tail $cur_playlist]]

# here is the main loop
set index 0
set wait_flag 0
while {1} {

  ## check first if we have any pictures to display
  while {[llength $files_list] == 0} {
    vwait files_list			;# wait for some files to be added
  }
  
  set image_file [index_to_filename $index]
  show_image $image_file
  
  follow_sel

  ## perform the countdown
  set secs_left $delay
  while {$secs_left > 0} {
    set delay_id [after 1000 {set wait_flag 1}]	;# wait for one second
    vwait wait_flag
    set wait_flag 0
    incr secs_left -1
    set time_left [secs_to_time_display $secs_left]
  }

  # if the image is locked, wait for the lock to be removed
  if {$image_lock} {
    vwait image_lock
  }
  
  incr index

  if {$index >= [$lb index end]} {
    set index 0
  }
}


